/*
 * --------------------------------------------------
 * ROR_GainPointByBattle.js
 *   ver.1.0.1
 * Copyright (c) 2024 R.Orio
 * This software is released under the MIT license.
 * http://opensource.org/licenses/mit-license.php
 * --------------------------------------------------
 */

/*:
 * @target MZ
 * @plugindesc 戦闘勝利時の報酬としてポイントを獲得する仕組みを提供します
 * @author R.Orio
 * @version 1.0.1
 * 
 * [version]
 * 1.0.0 初版
 * 1.0.1 倍率の値が0且つテストプレイの場合のみアラートを表示する機能を追加
 *
 *
 *
 * @param addTargetVariableId
 * @text ポイントを代入する変数ID
 * @desc 獲得したポイントを加算する変数のID
 * @default 0
 * @type variable
 * 
 * 
 * 
 * @param pointGainMessage
 * @text ポイント獲得時のメッセージ
 * @desc ポイントを獲得した際のメッセージを設定してください。文中の「%1」は獲得したポイント数に置換されます。
 * @default %1 ポイント獲得！
 * @type string
 * 
 * 
 * 
 * @param rateVariableId
 * @text ポイント獲得倍率が代入されている変数ID
 * @desc 獲得ポイント数に倍率をかける場合の倍率が代入されている変数のID
 * @default 0
 * @type variable
 * 
 * 
 * 
 * @param roundType
 * @text 小数点以下の処理
 * @desc 獲得ポイント数に端数が生じた場合の小数点以下の処理を選択
 * @type select
 * @option 切り上げ
 * @value 0
 * @option 四捨五入
 * @value 1
 * @option 切り捨て
 * @value 2
 * @default 0
 *
 *
 * @help
 * 戦闘勝利時に経験値やお金、アイテムとは別にポイントを獲得できます。
 * 獲得したポイントはプラグインパラメータで指定した変数に加算されます。
 * 
 * 獲得ポイント数は敵キャラのメモ欄に
 * <drop_point:xx>と記載することで設定できます。
 * （例：<drop_point:12>と記載した敵キャラを2体倒して
 * 　　　戦闘に勝利すると12×2＝24ポイント獲得）
 * 
 * 戦闘勝利時にポイントを獲得した旨のメッセージを表示する場合は
 * プラグインパラメータでメッセージを指定できます。
 * 文中に「%1」を含めるとその部分が獲得したポイント数に置き換わります。
 * 何も表示したくない場合は空欄で設定してください。
 * 
 * 獲得ポイント数に倍率をかけることも可能です。
 * 設定する場合はプラグインパラメータで変数IDを指定し、
 * その変数に%で値を代入してください。
 * ※整数で指定してください
 * （例：変数IDに10を指定し変数10番の値を110にしておくと
 * 　　　獲得ポイント数が1.1倍になります）
 * 
 * 尚、倍率は戦闘単位で計算します。敵キャラ単位ではないのでご了承ください。
 * 
 * 倍率を設定した場合は小数点以下の端数が生じた場合の処理方法も
 * プラグインパラメータで設定可能です。
 * デフォルトでは切り上げになっています。
 *
 *
 *
 * 利用規約:
 *   MITライセンスです。
 *   https://licenses.opensource.jp/MIT/MIT.html
 *   作者に無断で改変、再配布が可能で、
 *   利用形態（商用、18禁利用等）についても制限はありません。
 *
 */

(function () {
	'use strict';

	const pluginName = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
	const Parameters = PluginManager.parameters(pluginName);


	//獲得ポイントを計算するメソッドを定義
	Game_Troop.prototype.pointTotal = function(){
		let gain_point = 0;
		const Members = this.deadMembers();
		for(let Member of Members){
			if('_enemyId' in Member){
				if('drop_point' in $dataEnemies[Member._enemyId].meta){
					gain_point += parseInt($dataEnemies[Member._enemyId].meta.drop_point);
				}
			}
		}

		//倍率が指定されている場合は計算
		if(Parameters.rateVariableId > 0){
			if(parseInt($gameVariables.value(Parameters.rateVariableId)) == 0 && $gameTemp.isPlaytest()){
				const alert_text = "【ROR_GainPointByBattle.jsからのお知らせ】\n倍率として設定されている変数ID" + Parameters.rateVariableId + "の値が0のため\nポイントが付与されません。ご注意ください。";
				alert(alert_text);
			}

			if(Parameters.roundType == 1){
				//四捨五入
				gain_point = Math.round(gain_point * (parseInt($gameVariables.value(Parameters.rateVariableId)) / 100));
			}else if(Parameters.roundType == 2){
				//切り捨て
				gain_point = Math.floor(gain_point * (parseInt($gameVariables.value(Parameters.rateVariableId)) / 100));
			}else{
				//切り上げ
				gain_point = Math.ceil(gain_point * (parseInt($gameVariables.value(Parameters.rateVariableId)) / 100));
			}

			if(gain_point == 0 && $gameTemp.isPlaytest()){
				const alert_text = "【ROR_GainPointByBattle.jsからのお知らせ】\n倍率の値が小さすぎるため付与ポイントが0になっています。\n必要に応じて倍率の調整をご検討ください。";
				alert(alert_text);
			}
		}

		return gain_point;
	};


	//戦闘報酬を生成するメソッドにポイントを追加
	const _BattleManager_makeRewards = BattleManager.makeRewards;
	BattleManager.makeRewards = function(){
		_BattleManager_makeRewards.apply(this, arguments);
		this._rewards['point'] = $gameTroop.pointTotal();
	};


	//ポイントを獲得した旨を表示する文章を生成するメソッドを定義
	BattleManager.displayPoint = function(){
		const point = this._rewards.point;
		if(point > 0){
			const gain_text = Parameters.pointGainMessage;
			const TextParts = gain_text.split('%1');
			const text = TextParts[0] + point + TextParts[1];
			if(text.length > 0){
				$gameMessage.add("\\." + text);
			}
		}
	};


	//戦闘勝利時の獲得報酬のメッセージ表示メソッドにポイントの文章表示を追加
	const _BattleManager_displayRewards = BattleManager.displayRewards;
	BattleManager.displayRewards = function(){
		_BattleManager_displayRewards.apply(this, arguments);
		this.displayPoint();
	};


	//ポイント獲得処理を定義
	BattleManager.gainPoint = function(){
		const target_variable_id = Parameters.addTargetVariableId;
		const variable_value = parseInt($gameVariables.value(target_variable_id)) + this._rewards.point;
		$gameVariables.setValue(target_variable_id, variable_value);
	};


	//戦闘勝利時の報酬獲得処理にポイント獲得処理を追加
	const _BattleManager_gainRewards = BattleManager.gainRewards;
	BattleManager.gainRewards = function(){
		_BattleManager_gainRewards.apply(this, arguments);
		this.gainPoint();
	};
})();